/**
* \file: app_iface_abstract.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdlib.h>

#include "app_iface/app_iface_intern.h"
#include "app_iface/app_iface_extern.h"
#include "app_iface/socket_server.h"
#include "control/partition_fsm.h"
#include "control/device_fsm.h"
#include "model/partition_list.h"
#include "model/partition_metadata.h"
#include "model/device_list.h"
#include "model/device_metadata.h"
#include "utils/logger.h"

static const app_iface_event_callbacks_t *app_iface_event_callbacks=NULL;

error_code_t app_iface_init(void)
{
	error_code_t result;

	logger_log_debug("APP-IFACE - Initializing the application interface.");
	result=socket_server_init(app_iface_handle_message);

	if (result==RESULT_OK)
		result=app_iface_init_impl();

	return result;
}

void app_iface_deinit(void)
{
	app_iface_deinit_impl();
	socket_server_deinit();
}

void app_iface_signal_device_detected(device_t *device)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_device_detected!=NULL)
		app_iface_event_callbacks->on_device_detected(device);
}

void app_iface_signal_device_nomedia(device_t *device)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_device_nomedia!=NULL)
		app_iface_event_callbacks->on_device_nomedia(device);
}

void app_iface_signal_device_automounting(device_t *device)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_device_automounting!=NULL)
		app_iface_event_callbacks->on_device_automounting(device);
}

void app_iface_signal_device_automounted(device_t *device)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_device_automounted!=NULL)
		app_iface_event_callbacks->on_device_automounted(device);
}

void app_iface_signal_device_unmounting(device_t *device)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_device_unmounting!=NULL)
		app_iface_event_callbacks->on_device_unmounting(device);
}

void app_iface_signal_device_unmounted(device_t *device)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_device_unmounted!=NULL)
		app_iface_event_callbacks->on_device_unmounted(device);
}

void app_iface_signal_device_invalid(device_t *device)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_device_invalid!=NULL)
		app_iface_event_callbacks->on_device_invalid(device);
}

void app_iface_signal_partition_detected(partition_t *partition)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_partition_detected!=NULL)
		app_iface_event_callbacks->on_partition_detected(partition);
}

void app_iface_signal_partition_unsupported(partition_t *partition)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_partition_unsupported!=NULL)
		app_iface_event_callbacks->on_partition_unsupported(partition);
}

void app_iface_signal_partition_mounting(partition_t *partition)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_partition_mounting!=NULL)
		app_iface_event_callbacks->on_partition_mounting(partition);
}

void app_iface_signal_partition_mounted(partition_t *partition)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_partition_mounted!=NULL)
		app_iface_event_callbacks->on_partition_mounted(partition);
}

void app_iface_signal_partition_mount_err(partition_t *partition)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_partition_mount_err!=NULL)
		app_iface_event_callbacks->on_partition_mount_err(partition);
}

void app_iface_signal_partition_remounting(partition_t *partition)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_partition_remounting!=NULL)
		app_iface_event_callbacks->on_partition_remounting(partition);
}

void app_iface_signal_partition_unmounting(partition_t *partition)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_partition_unmounting!=NULL)
		app_iface_event_callbacks->on_partition_unmounting(partition);
}

void app_iface_signal_partition_unmounted(partition_t *partition)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_partition_unmounted!=NULL)
		app_iface_event_callbacks->on_partition_unmounted(partition);
}

void app_iface_signal_partition_invalid(partition_t *partition)
{
	if (app_iface_event_callbacks==NULL)
		return;
	if (app_iface_event_callbacks->on_partition_invalid!=NULL)
		app_iface_event_callbacks->on_partition_invalid(partition);
}

error_code_t app_iface_unmount_mountpoint_request(const char *mount_point,
        partition_state_change_callback_t callback_func,app_iface_request_ctx_t *request_ctx)
{
    partition_t *partition;
    partition=device_list_find_partition_by_mountpoint(mount_point);

    if (partition!=NULL)
    {
        return partition_fsm_signal_unmount_request(partition,callback_func,request_ctx);
    }
    else
        return RESULT_INVALID;
}

error_code_t app_iface_unmount_partition_request(const char *partition_id,
        partition_state_change_callback_t callback_func,app_iface_request_ctx_t *request_ctx)
{
    partition_t *partition;
    partition=device_list_find_partition_by_id(partition_id);

    if (partition!=NULL)
    {
        return partition_fsm_signal_unmount_request(partition,callback_func,request_ctx);
    }
    else
        return RESULT_INVALID;
}

error_code_t app_iface_unmount_device_request(const char *device_id,
        device_state_change_callback_t callback_func,app_iface_request_ctx_t *request_ctx)
{
    device_t *device;
    device=device_list_find_device(device_id);

    if (device!=NULL)
        {
            return device_fsm_signal_device_unmount_request(device,callback_func,request_ctx);
        }
    else
        return RESULT_INVALID;
}

error_code_t app_iface_remount_request_mp(const char *mount_point, const char *mount_options,
		partition_state_change_callback_t callback_func,app_iface_request_ctx_t *request_ctx)
{
	partition_t *partition;
	mount_options_t options={0, mount_options};

	logger_log_debug("APP_IFACE - Remount request (id=%d) for a partition received from application '%s':",
			request_ctx->request_id, socket_server_get_connection_appidentifier(request_ctx->ctx));
	logger_log_debug("		Mount Point:   %s",mount_point);
	logger_log_debug("		Mount options: %s", mount_options);

	partition=device_list_find_partition_by_mountpoint(mount_point);
	if (partition!=NULL)
		return partition_fsm_signal_remount_request(partition, &options,callback_func,request_ctx);
	else
		return RESULT_INVALID;
}

error_code_t app_iface_remount_request_id(const char *partition_id, const char *mount_options,
		partition_state_change_callback_t callback_func,app_iface_request_ctx_t *request_ctx)
{
	partition_t *partition;
	mount_options_t options={0, mount_options};

	logger_log_debug("APP_IFACE - Remount request (id=%d) for a partition received from application '%s':",
			request_ctx->request_id, socket_server_get_connection_appidentifier(request_ctx->ctx));
	logger_log_debug("		Partition Id:   %s",partition_id);
	logger_log_debug("		Mount options: %s", mount_options);

	partition=device_list_find_partition_by_id(partition_id);
	if (partition!=NULL)
		return partition_fsm_signal_remount_request(partition, &options,callback_func,request_ctx);
	else
		return RESULT_INVALID;
}

void app_iface_register_event_callbacks(const app_iface_event_callbacks_t *event_callbacks)
{
	app_iface_event_callbacks=event_callbacks;
}

void app_iface_extract_part_info(partition_info_t *part_info,partition_t *partition)
{
	part_info->data_structure_real_size=sizeof(partition_info_t);
	part_info->extension_bitmask=0;
	part_info->error=partition_get_error(partition);
	part_info->identifier=partition_metadata_get_partition_identifier(partition);
	part_info->interface_id=partition_get_id(partition);
	part_info->mount_fs=partition_metadata_get_mountfs(partition);
	part_info->mount_src=partition_metadata_get_mountsrc(partition);
	part_info->mount_point=partition_get_mountpoint(partition);
	part_info->mounted_writable=partition_is_mounted_writable(partition);
	part_info->partition_no=partition_metadata_get_partition_number(partition);
	part_info->state=partition_get_state(partition);
	part_info->unsupported_reason=partition_get_unsupported_reason(partition);
}

void app_iface_extract_dev_info(device_info_t *device_info,device_t *device)
{
	device_info->data_structure_real_size=sizeof(device_info_t);
	device_info->extension_bitmask=0;
	device_info->detected_partition_cnt=device_get_detected_partition_cnt(device);
	device_info->device_handler_id=device_get_handler_id(device);
	device_info->device_type=device_get_device_type(device);
	device_info->identifier=device_metadata_get_device_identifier(device);
	device_info->interface_id=device_get_id(device);
	device_info->state=device_get_state(device);
}

